/**************************************************************/
/* nc2emc                                                     */
/* ---------------------------------------------------------- */
/* Version:  V 1.0                                            */
/* Date:     04/29/2005                                       */
/* Author:   Michael Schlagmueller                            */
/* Email:    emc [at] schlagmueller [dot] com                 */
/* Web:      http://www.schlagmueller.com                     */
/* Company:  DLR Stuttgart                                    */
/* Web:      http://www.eid.dlr.de/dlr/;internal&LANGUAGE=en  */
/* ---------------------------------------------------------  */
/* nc2emc is a fontend for hp2xx to create usable g-code      */
/* for EMC (included in the BDI linux distribution            */
/* ---------------------------------------------------------  */
/* hp2xx     www.gnu.org/software/hp2xx/hp2xx.html  V 3.4.4   */
/* EMC       www.linuxcnc.org                                 */
/* BDI       www.linuxcnc.org/bdi                   V 4.08    */
/* ---------------------------------------------------------  */
/* comment                                                    */
/*                                                            */
/* You may use and modify this program the way you want to.   */
/* There is only one think you should do: please keep my      */
/* name anywhere in this file. If you have any questions or   */
/* if you do any improvement I would be glad to receive a     */
/* small litte note.                                          */
/* There is absolutely no warranty for nc2emc                 */
/* ---------------------------------------------------------  */
/* platforms notes                                            */
/* win32     copy tclap in your include directory or adjust   */
/*           you include path for this application            */
/* linux     gpp nc2emc.cpp -I .                              */
/**************************************************************/

#include <string>
#include <iostream>
#include <algorithm>
#include <tclap/CmdLine.h>
#include <fstream>
#include <sstream>

using namespace TCLAP;
using namespace std;


class Pars {
public:
    string fileName;
    double depth;
	double height;
	double speedI;
	double speedO;
	double scale;
	double moveX;
	double moveY;
	bool inch;
	bool center;
	bool centerX;
	bool centerY;
	bool preview;
}; 

class CenterPar {
private:
	double DOUBLE_UNDEF;
public:
	double maxX;
	double minX;
	double minY;
	double maxY;
	CenterPar(){
		clear();
	}
	void clear() {
		DOUBLE_UNDEF = 0.000000001;
		maxX = DOUBLE_UNDEF;
		minX = DOUBLE_UNDEF;
		minY = DOUBLE_UNDEF;
		maxY = DOUBLE_UNDEF;
		enabledX = false;
		enabledY = false;
	}
	
	bool enabledX;
	bool enabledY;

	double getMovX() {
		if ((maxX == DOUBLE_UNDEF) || (enabledX == false)) {
			return 0;
		} else {
			return ((maxX - minX) / 2.0);
		}
	}
	
	double getMovY(){
		if ((maxY == DOUBLE_UNDEF) || (enabledY == false)) {
			return 0;
		} else {
			return ((maxY - minY) / 2.0);
		}
	}

	void newX(double x) {
		if (minX == DOUBLE_UNDEF) {
			minX = x;
			maxX = x;
		} else {
			minX = min(x, minX);
			maxX = max(x, maxX);
		}
	}

	void newY(double y) {
		if (minY == DOUBLE_UNDEF) {
			minY = y;
			maxY = y;
		} else {
			minY = min(y, minY);
			maxY = max(y, maxY);
		}
	}

};

Pars pars;
CenterPar centerPar;
CenterPar minmaxCoord;


void clearStream(ifstream* inStream, stringstream* outStream);
void optimizeStream(stringstream* inStream, stringstream* outStream);
void readStream(stringstream* inStream, stringstream* outStream);
void ncStream(stringstream* inStream, ofstream* outStream);

template <class A, class B>
void split2(string zeile, A *a, B *b)
{
	stringstream ss;
	ss << zeile;
	ss >> *a;
	ss >> *b;
}


template <class A, class B, class C, class D, class E>
void split5(string zeile, A *a, B *b, C *c, D *d, E *e) {
	stringstream ss;
	ss << zeile;
	ss >> *a;
	ss >> *b;
	ss >> *c;
	ss >> *d;
	ss >> *e;
}

template <class T>
void string_fmt(string & val, const T & t)
{
 ostringstream oss; // create a stream
 oss << t; // insert value to stream 
 val=oss.str(); // extract string and copy 
}

template <class out_type, class in_value>
out_type cast_stream(const in_value & t)
{
 stringstream ss;
 ss << t; // first insert value to stream
 out_type result; // value will be converted to out_type
 ss >> result; // write value to result
 return result;
}


void openerror( const string datei)
{
  cerr << "Fehler beim ffnen von " << datei << endl;
  exit(1);
};

int main(int argc, char** argv)
{
	cout << "nc2emc V 1.0 (04/29/2005)" << endl;
	cout << "Author: Michael Schlagmueller - ";
	cout << "http://www.schlagmueller.com - ";
	cout << "emc [at] schlagmueller [dot] com - ";
	cout << "Company: ";
	cout << "DLR Stuttgart - ";
	cout << "http://www.eid.dlr.de/dlr/;internal&LANGUAGE=en" << endl;
	cout << endl;
	cout << endl;

	// Wrap everything in a try block.  Do this every time, 
	// because exceptions will be thrown for problems. 
	try {  

		// Define the command line object.
		CmdLine cmd("Command description message", ' ', "0.9");

		// Define a value argument and add it to the command line.
		UnlabeledValueArg<string> fileArg("fileName","file name", "","filename");
		cmd.add( fileArg );

		ValueArg<double> depthArg("","depth","working depth",false,5.0,"float");
		cmd.add( depthArg );

		ValueArg<double> heightArg("","distance","height above workpiece",false,10.0,"float");
		cmd.add( heightArg );

		ValueArg<double> speedIArg("","working_speed","speed while in workpiece",false,10,"float");
		cmd.add( speedIArg );

		ValueArg<double> speedOArg("","driving_speed","speed above in workpiece",false,600,"float");
		cmd.add( speedOArg );

		ValueArg<double> scaleArg("","scale","scale whole plot",false,1.0,"float");
		cmd.add( scaleArg );

		ValueArg<double> moveYArg("","move_y","move output in y direction",false,0.0,"float");
		cmd.add( moveYArg );

		ValueArg<double> moveXArg("","move_x","move output in x direction",false,0.0,"float");
		cmd.add( moveXArg );

		// Define a switch and add it to the command line.
		SwitchArg inchSwitch("i","inch","system selection", false);
		cmd.add( inchSwitch );

		SwitchArg previewSwitch("p","preview","create preview picture (wrong pensize - be careful)", false);
		cmd.add( previewSwitch );

		SwitchArg centerYSwitch("","center_y","center plot in y-direction", false);
		cmd.add( centerYSwitch );

		SwitchArg centerXSwitch("","center_x","center plot in x-direction", false);
		cmd.add( centerXSwitch );

		SwitchArg centerSwitch("c","center","center plot in both directions", false);
		cmd.add( centerSwitch );


		// Parse the args.
		cmd.parse( argc, argv );

		pars.fileName = fileArg.getValue();
		pars.depth = depthArg.getValue();
		pars.height = heightArg.getValue();
		pars.speedI = speedIArg.getValue();
		pars.speedO = speedOArg.getValue();
		pars.scale = scaleArg.getValue();
		pars.moveX = moveXArg.getValue();
		pars.moveY = moveYArg.getValue();
		pars.inch = inchSwitch.getValue();
		pars.preview = previewSwitch.getValue();
		pars.center = centerSwitch.getValue();
		pars.centerX = centerXSwitch.getValue();
		pars.centerY = centerYSwitch.getValue();

		if (pars.center) {
			centerPar.enabledX = true;
			centerPar.enabledY = true;
		}
		if (pars.centerX) {
			centerPar.enabledX = true;
		}
		if (pars.centerY) {
			centerPar.enabledY = true;
		}

		cout << "> start converting '" << pars.fileName << "' <" << endl;
		cout << "preview:" << endl;
		string systemcall;
		string quelle = pars.fileName;
		string ziel = quelle + ".emc";
		systemcall = "hp2xx -q " + quelle;
		if (system( systemcall.c_str() ) != 0) {
			cerr << "FAILURE ! External call failed !" << endl;
			return 1;
		};


		systemcall = "hp2xx -q -t -m nc " + quelle + " -f " + quelle + ".nc";
		cout << "external call of '" << systemcall << "'" << endl << endl;
		if (system( systemcall.c_str() ) != 0) {
			cerr << "FAILURE ! External call failed !" << endl;
			return 1;
		};

		cout << "> start converting '" << pars.fileName << "' <" << endl;

		string inFile;
		inFile = quelle + ".nc";
		ifstream hpglFile( inFile.c_str(), ios::in ); 
		if( !hpglFile ) {
			openerror( inFile );
			return 1;
		}
		// check overwrite missing !!!
		ofstream outFile( ziel.c_str(), ios::out ); 
		if( !outFile ) {
			openerror( ziel );
			hpglFile.close();
			return 1;
		}

		stringstream postClean;
		cout << "-cleanup comments and unnecessary lines" << endl;
		clearStream(&hpglFile, &postClean);

		stringstream postOptimize;
		cout << "-optimze" << endl;
		optimizeStream(&postClean, &postOptimize);

		stringstream postRead;
		cout << "-read" << endl;
		readStream(&postOptimize, &postRead);

		cout << "-hpgl2nc" << endl;
		ncStream(&postRead, &outFile);

		cout << "> done <" << endl;
		cout << endl;
		cout << "Output File: '" << ziel << "'" << endl;
		cout << "Dimentions ";
		
		if (pars.inch) {
			cout << "(inch)";
		} else {
			cout << "(mm)";
		}
		cout << "  X: " << minmaxCoord.minX << " to " << minmaxCoord.maxX << "  Y: " << minmaxCoord.minY << " to " << minmaxCoord.maxY << "  Z: -" << pars.depth << " to " << pars.height << endl;
		cout << endl;
		cout << "Ready... Have a nice day !" << endl;
	} catch (ArgException &e)  // catch any exceptions
	{ cerr << "error: " << e.error() << " for arg " << e.argId() << endl; }
}

void clearStream(ifstream* inStream, stringstream* outStream) {
	string line;
	while (getline(*inStream, line)) {
		if ((line.length() == 0) || (line[0] != 'G')) continue;
		*outStream << line << endl;
	}
}

void readStream(stringstream* inStream, stringstream* outStream) {
	string line;
	string firstLine;
	string GCode, Dir1, Dir2;
	double Pos1, Pos2;
	if (getline(*inStream, firstLine)) {
		*outStream << firstLine << endl;
		while (getline(*inStream, line) )
		{
			*outStream << line << endl;
			if (line.c_str()[4] == 'Z')  {
				continue;
			} else {
				split5(line, &GCode, &Dir1, &Pos1, &Dir2, &Pos2);
				centerPar.newX(Pos1);
				centerPar.newY(Pos2);
				continue;
			}
		}
		*outStream << firstLine << endl;
	}
}

void optimizeStream(stringstream* inStream, stringstream* outStream) {
	string s0;
	string s1;
	string s2;
	string s3;

	int skipwrite = 3;
	bool write;
	string line;
	
	while (getline(*inStream, line) )
	{
		write = true;

		s3 = s2;
		s2 = s1;
		s1 = s0;
		s0 = line;

		if (skipwrite == 0) {
			if ((s2[4] == 'Z') && (s0[4] == 'Z')) {
				if (s1.substr(3).compare(s3.substr(3)) == 0) {
					skipwrite = 4;
					cout << "x";
				}
			}
		}

		if (skipwrite > 0) {
			skipwrite--;
			write = false;
			continue;
		}
		
		if (write) *outStream << s3 << endl;
	}
	if (skipwrite > 0) {
		skipwrite--;
	} else {
		*outStream << s2 << endl;
	}
	if (skipwrite > 0) {
		skipwrite--;
	} else {
		*outStream << s1 << endl;
	}
	if (skipwrite > 0) {
		skipwrite--;
	} else {
		*outStream << s0 << endl;
	}
	cout << endl;
}


void ncStream(stringstream* inStream, ofstream* outStream) {
	// HEADER
	*outStream << "G40" << endl; // disable cutter compensation
	if (pars.inch) {
		*outStream << "G20" << endl; // set inch system
	} else {
		*outStream << "G21" << endl; // set millimeter system
	}

	//MAINCODE

	string line;
	double Pos1, Pos2;
	string GCode, Par, Dir1, Dir2;
	bool up = true;
	while (getline(*inStream, line) )
	{
		if (line.c_str()[4] == 'Z')  {
			split2(line, &GCode, &Par);


			if (Par[1] == '-') {
				up = false;
			} else up = true;

			if (up) {
				*outStream << GCode << " Z";
				*outStream <<          pars.height << " F" << pars.speedO << endl;
			} else {
				*outStream << GCode << " Z";
				*outStream <<          "0.2" << " F" << pars.speedO << endl;
				*outStream << GCode << " Z";
				*outStream << "-"   << pars.depth << " F" << pars.speedI << endl;
			}
			
			continue;
		}

		split5(line, &GCode, &Dir1, &Pos1, &Dir2, &Pos2);

		Pos1 -= centerPar.getMovX();
		Pos1 += pars.moveX;
		Pos1 *= pars.scale;
		
		Pos2 -= centerPar.getMovY();
		Pos2 += pars.moveY;
		Pos2 *= pars.scale;

		minmaxCoord.newX(Pos1);
		minmaxCoord.newY(Pos2);

		*outStream << GCode << " " << Dir1 << " " << Pos1 << " " << Dir2 << " " << Pos2;
		
		if (up) {
			*outStream << " F" << pars.speedO << endl;
		} else {
			*outStream << " F" << pars.speedI << endl;
		}

	}

	//FOOTER
	*outStream << "%" << endl;
}
